import React, { useEffect, useState } from "react";
import axios from "axios";
import { Link } from "react-router-dom";
import { FaHeart } from "react-icons/fa";
import { IoMdHeartDislike } from "react-icons/io";

export default function Favorites() {
  const [favorites, setFavorites] = useState([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    setLoading(true);
    axios
      .get("http://localhost:5000/api/movies/favorites")
      .then((res) => {
        setFavorites(res.data);
        setLoading(false);
      })
      .catch((err) => {
        console.error(err);
        setLoading(false);
      });
  }, []);

  const removeFavorite = async (id) => {
    try {
      await axios.delete(`http://localhost:5000/api/movies/favorites/${id}`);
      setFavorites(favorites.filter((f) => f._id !== id));
    } catch (err) {
      console.error(err);
      alert("Failed to remove from favorites");
    }
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-rose-900 via-pink-900 to-purple-900">
      <div className="container mx-auto px-6 py-12">
        {/* Header */}
        <div className="text-center mb-12">
          <h1 className="text-6xl font-bold text-black mb-4 bg-gradient-to-r from-rose-400 to-pink-400 bg-clip-text text-transparent">
            Favorite Movies
          </h1>
          <p className="text-gray-300 text-lg">
            Your personal collection of beloved films
          </p>
        </div>

        {/* Loading State */}
        {loading && (
          <div className="flex justify-center items-center py-20">
            <div className="animate-spin rounded-full h-16 w-16 border-t-4 border-b-4 border-rose-500"></div>
          </div>
        )}

        {/* Favorites Grid */}
        {!loading && favorites.length > 0 && (
          <div>
            <div className="flex items-center justify-between mb-6">
              <h2 className="text-2xl font-semibold text-black">
                {favorites.length} {favorites.length === 1 ? "Movie" : "Movies"}
              </h2>
            </div>
            <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 xl:grid-cols-5 gap-6">
              {favorites.map((f) => (
                <div
                  key={f._id}
                  className="group relative bg-white bg-opacity-10 backdrop-blur-lg rounded-2xl overflow-hidden border border-white border-opacity-20 shadow-xl hover:shadow-2xl transition-all duration-300 transform hover:scale-105"
                >
                  {/* Movie Poster */}
                  <div className="relative overflow-hidden">
                    <img
                      src={
                        f.poster !== "N/A"
                          ? f.poster
                          : "https://via.placeholder.com/300x450?text=No+Image"
                      }
                      alt={f.title}
                      className="w-full h-80 object-cover transition-transform duration-500 group-hover:scale-110"
                    />
                    {/* Overlay on Hover */}
                    <div className="absolute inset-0 bg-gradient-to-t from-black via-transparent to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>

                    {/* Remove Button */}
                    <button
                      onClick={() => removeFavorite(f._id)}
                      className="absolute top-3 right-3 bg-rose-500 text-black p-2 rounded-full opacity-0 group-hover:opacity-100 transition-opacity duration-300 hover:bg-rose-600 shadow-lg"
                      title="Remove from favorites"
                    >
                      <svg
                        xmlns="http://www.w3.org/2000/svg"
                        className="h-5 w-5"
                        viewBox="0 0 20 20"
                        fill="currentColor"
                      >
                        <path
                          fillRule="evenodd"
                          d="M4.293 4.293a1 1 0 011.414 0L10 8.586l4.293-4.293a1 1 0 111.414 1.414L11.414 10l4.293 4.293a1 1 0 01-1.414 1.414L10 11.414l-4.293 4.293a1 1 0 01-1.414-1.414L8.586 10 4.293 5.707a1 1 0 010-1.414z"
                          clipRule="evenodd"
                        />
                      </svg>
                    </button>
                  </div>

                  {/* Movie Title */}
                  <div className="p-4">
                    <h3 className="text-black font-semibold text-lg line-clamp-2">
                      {f.title}
                    </h3>
                    {f.year && (
                      <p className="text-gray-300 text-sm mt-1">{f.year}</p>
                    )}
                  </div>

                  {/* Favorite Badge */}
                  <div className="absolute top-3 left-3 bg-rose-500 text-black p-2 rounded-full text-xl font-semibold shadow-lg">
                    <FaHeart />
                  </div>
                </div>
              ))}
            </div>
          </div>
        )}

        {/* Empty State */}
        {!loading && favorites.length === 0 && (
          <div className="text-center py-20">
            
            <h2 className="text-3xl font-bold text-white mb-4">
              No favorites yet
            </h2>
            <p className="text-gray-300 text-lg mb-8">
              Start adding movies to your favorites to see them here!
            </p>

            <Link to='/' className="inline-block px-8 py-4 bg-gradient-to-r from-rose-500 to-pink-600 text-white font-semibold rounded-xl hover:from-rose-600 hover:to-pink-700 transition-all duration-300 transform hover:scale-105 shadow-lg">
              Discover Movies
            </Link>
          </div>
        )}
      </div>
    </div>
  );
}
